
/**
 * EException.h - (c) EliteMMO Network, 2015
 *
 * This unpublished material is proprietary to EliteMMO Network.
 * All rights reserved.
 *
 * The methods and techniques described herein are considered trade secrets
 * and/or condifential. Reproduction or distribution, in whole or in part,
 * is forbidden except by express written permission of EliteMMO Network.
 *
 * This file is part of the EliteMMO Development Kit.
 */

#ifndef __EDK_EXCEPTION_H_INCLUDED__
#define __EDK_EXCEPTION_H_INCLUDED__

#if defined (_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#include <Windows.h>
#include <eh.h>

namespace EliteMMO
{
    namespace Exception
    {
        /**
         * @brief Define misssing STATUS code.
         */
#ifndef STATUS_POSSIBLE_DEADLOCK
#define STATUS_POSSIBLE_DEADLOCK 0xC0000194
#endif

        /**
         * @brief Macro for translating exception defines to strings.
         */
#define CASE(Exception) \
    case Exception: \
        this->m_Exception = #Exception; \
        break;

        /**
         * @brief Exception object that can obtain all exceptions.
         */
        class EException
        {
            unsigned int    m_ExceptionId;
            const char*     m_Exception;

        public:
            /**
             * @brief Constructor
             */
            EException(unsigned int id)
                : m_ExceptionId(id)
            {
                switch (this->m_ExceptionId)
                {
                    CASE(EXCEPTION_ACCESS_VIOLATION);
                    CASE(EXCEPTION_ARRAY_BOUNDS_EXCEEDED);
                    CASE(EXCEPTION_BREAKPOINT);
                    CASE(EXCEPTION_DATATYPE_MISALIGNMENT);
                    CASE(EXCEPTION_FLT_DENORMAL_OPERAND);
                    CASE(EXCEPTION_FLT_DIVIDE_BY_ZERO);
                    CASE(EXCEPTION_FLT_INEXACT_RESULT);
                    CASE(EXCEPTION_FLT_INVALID_OPERATION);
                    CASE(EXCEPTION_FLT_OVERFLOW);
                    CASE(EXCEPTION_FLT_STACK_CHECK);
                    CASE(EXCEPTION_FLT_UNDERFLOW);
                    CASE(EXCEPTION_GUARD_PAGE);
                    CASE(EXCEPTION_ILLEGAL_INSTRUCTION);
                    CASE(EXCEPTION_IN_PAGE_ERROR);
                    CASE(EXCEPTION_INT_DIVIDE_BY_ZERO);
                    CASE(EXCEPTION_INT_OVERFLOW);
                    CASE(EXCEPTION_INVALID_DISPOSITION);
                    CASE(EXCEPTION_INVALID_HANDLE);
                    CASE(EXCEPTION_NONCONTINUABLE_EXCEPTION);
                    CASE(EXCEPTION_POSSIBLE_DEADLOCK);
                    CASE(EXCEPTION_PRIV_INSTRUCTION);
                    CASE(EXCEPTION_SINGLE_STEP);
                    CASE(EXCEPTION_STACK_OVERFLOW);

                default:
                    this->m_Exception = "Unknown exception occurred.";
                    break;
                }
            }

            /**
             * @brief Deconstructor
             */
            ~EException(void)
            { }

        public:
            /**
             * @brief Returns the exception id of this exception object.
             */
            unsigned int GetExceptionId(void) const
            {
                return this->m_ExceptionId;
            }

            /**
             * @brief Returns the exception string of this exception object.
             */
            const char* GetException(void) const
            {
                return this->m_Exception;
            }
        };

        /**
         * @brief Scoped SEH translator class to automate custom exception filtering.
         */
        class ScopedTranslator
        {
            _se_translator_function m_Function;

        public:
            /**
             * @brief Constructor
             */
            ScopedTranslator(void)
            {
                this->m_Function = ::_set_se_translator(&ScopedTranslator::ScopedTranslatorFunc);
            }

            /**
             * @brief Deconstructor
             */
            ~ScopedTranslator(void)
            {
                if (this->m_Function != NULL)
                {
                    ::_set_se_translator(this->m_Function);
                    this->m_Function = NULL;
                }
            }

        private:
            /**
             * @brief Exception filter.
             */
            static void ScopedTranslatorFunc(unsigned int id, struct _EXCEPTION_POINTERS* lpPtrs)
            {
                // Rethrow this exception with our wrapper..
                UNREFERENCED_PARAMETER(lpPtrs);
                throw EliteMMO::Exception::EException(id);
            }
        };
    }; // namespace Exception
}; // namespace EliteMMO

#endif // __EDK_EXCEPTION_H_INCLUDED__
